*-------------------------------------------------------
* Function....: RepairDatabase
* Called by...:
* Abstract....:
*
* Returns.....:
*
* Parameters..: tcPath
*
* Notes.......: Repairs a broken DBC.
*-------------------------------------------------------
FUNCTION RepairDatabase
LPARAMETERS tcPath

LOCAL lnErrorNo, lcDBCPath, llDBCIsOpen, loException, lnMaxItems, lnFileNumber, lcTempDatabaseDirectory, ;
			lcFrom, lcTo, lnNumAddedTables, lnFilesCount, lnCurrentFile, lcTableName, lcTableNameCDX, lcTableNameFPT, ;
			llCopyTable, lnNumAddedTables, lcDBCDirecotoryOriginal, lnItem, lcCurrentTable, llRepairDBF, llRepairCDX, ;
			llRepairFPT, lnMaxErrorCount, lcBaseDBF, lcBaseDBFCDX, lcBaseDBFFPT, lnFileHandleNumber1, lnFileHandleNumber2, ;
			llReplaceMemoFile, lnFieldCount, lcDropMemo, lnCurrentField, lnI, lnRow, lcNotRepairTables, lcComma, ;
			lnTableCount, lcCurrTable, lnCurrentRow, lnLastAutoIncValue, lcFieldName, lcBuffer, llDBCNotValid, ;
			lcDatabase 

DIMENSION laAutoIncs[1, 4]
DIMENSION laTable[1,1]

IF EMPTY(tcPath)	
	MESSAGEBOX("Please pass a path.", 16, _Screen.Caption)
	RETURN .F.
ENDIF 

STORE "" TO lcNotRepairTables, lcComma, lcDBCPath

WAIT WINDOW 'Database repair process is running...' NOWAIT NOCLEAR 

lcDatabase = FORCEEXT(GetDbcName(), 'dbc')
lcDatabaseName = FORCEEXT(JUSTFNAME(lcDatabase), "dbc")

* 1. 
lcBaseDirectory = ADDBS(ADDBS(SYS(2023)) + SYS(2015))
MD (lcBaseDirectory)

IF !CreateEmptyDBC(lcDatabaseName, lcBaseDirectory)
	WAIT CLEAR 
	MESSAGEBOX("Empty database cannot be created. Operation canceled.", 0 + 16, _Screen.Caption)
	RETURN .F.
ENDIF

* 2. 
lnFileNumber = ADIR(laAllFiles, ADDBS(tcPath) + "*.*", "A")
lnErrorNo = 0

lcTempDatabaseDirectory = ADDBS(ADDBS(SYS(2023)) + SYS(2015))
MD (lcTempDatabaseDirectory)

TRY
	CLOSE DATABASES ALL
	FOR lnKk = 1 TO lnFileNumber
		IF USED(JUSTSTEM(laAllFiles[lnKk, 1])) OR ;
			INLIST(LOWER(JUSTEXT(laAllFiles[lnKk, 1])), "fxp", "prg") OR ;
			(alltrim(lower(juststem(laAllFiles[lnKk, 1]))) == 'vfx_' + lower(juststem(lcDatabase)) + "krt")
			LOOP 
		ENDIF

		COPY FILE (ADDBS(tcPath) + laAllFiles[lnKk, 1]) TO (lcTempDatabaseDirectory + laAllFiles[lnKk, 1])
	ENDFOR
CATCH TO loError
	lnErrorNo = loError.ErrorNo
ENDTRY

IF lnErrorNo > 0
	DelDirectory(lcTempDatabaseDirectory)
	WAIT CLEAR 
	MESSAGEBOX("Files cannot be copied to '" + alltrim(lcTempDatabaseDirectory) + "'. Operation canceled.", 0 + 16, _Screen.Caption)
	RETURN .F.
ENDIF



* 3. 
* Entire database
* The database must be opened exclusive.
lcDBCPath = ADDBS(lcTempDatabaseDirectory) + lcDatabase

TRY
	OPEN DATABASE (lcDBCPath) EXCLUSIVE
	SET DATABASE TO (lcDBCPath)
CATCH TO loException 
	lnErrorNo = loException.ErrorNo
ENDTRY

IF lnErrorNo > 0
	llDBCNotValid = .T.
ENDIF

IF DBUSED(lcDatabase)
	SET DATABASE TO (lcDatabase)
	CLOSE DATABASES
ENDIF
lnErrorNo = 0

* 4.
* Copy valid DBC files over the broken DBC folder
IF llDBCNotValid 
	TRY
		lcFrom = lcBaseDirectory + lcDatabaseName
		lcTo = lcTempDatabaseDirectory + lcDatabaseName
		COPY FILE (FORCEEXT(lcFrom, "dbc")) TO (FORCEEXT(lcTo, "dbc"))
		COPY FILE (FORCEEXT(lcFrom, "dct")) TO (FORCEEXT(lcTo, "dct"))
		COPY FILE (FORCEEXT(lcFrom, "dcx")) TO (FORCEEXT(lcTo, "dcx"))
	CATCH
	ENDTRY
ENDIF

* 5.
* Check for deleted tables
lnNumAddedTables = 0
lnFilesCount = ADIR(laAllNewTables, lcBaseDirectory + "*.dbf")
FOR lnCurrentFile = 1 TO lnFilesCount
	lcTableName = laAllNewTables[lnCurrentFile, 1]
	lcTableNameCDX = FORCEEXT(lcTableName, "cdx")
	lcTableNameFPT = FORCEEXT(lcTableName, "fpt")
	
	llCopyTable = .F.
	TRY
		IF ADIR(laDummy, lcBaseDirectory + lcTableName) = 1 AND ADIR(laDummy, lcTempDatabaseDirectory + lcTableName) = 0
			COPY FILE (lcBaseDirectory + lcTableName) TO (lcTempDatabaseDirectory + lcTableName)
			llCopyTable = .T.
			IF ADIR(laDummy, lcBaseDirectory + lcTableNameCDX) = 1 AND ADIR(laDummy, lcTempDatabaseDirectory + lcTableNameCDX) = 0
				COPY FILE (lcBaseDirectory + lcTableNameCDX) TO (lcTempDatabaseDirectory + lcTableNameCDX)
			ENDIF
			IF ADIR(laDummy, lcBaseDirectory + lcTableNameFPT) = 1 AND ADIR(laDummy, lcTempDatabaseDirectory + lcTableNameFPT) = 0
				COPY FILE (lcBaseDirectory + lcTableNameFPT) TO (lcTempDatabaseDirectory + lcTableNameFPT)
			ENDIF							
		ENDIF
	CATCH TO loError
		* Do Nothing
		llCopyTable = .F.
	ENDTRY
ENDFOR

* 6.	
lcDBCDirecotoryOriginal = ADDBS(ADDBS(SYS(2023)) + SYS(2015))
TRY
	MD (lcDBCDirecotoryOriginal)
	lcFrom = lcBaseDirectory + lcDatabaseName
	lcTo = lcDBCDirecotoryOriginal + lcDatabaseName
	COPY FILE (lcFrom) TO (lcTo)
	COPY FILE (FORCEEXT(lcFrom, "dct")) TO (FORCEEXT(lcTo, "dct"))
	COPY FILE (FORCEEXT(lcFrom, "dcx")) TO (FORCEEXT(lcTo, "dcx"))
CATCH TO loError
	lnErrorNo = loError.ErrorNo
ENDTRY

IF lnErrorNo > 0
	WAIT CLEAR 
	MESSAGEBOX("Files cannot be copied to '" + alltrim(lcDBCDirecotoryOriginal) + "'. Operation canceled.", 0 + 16, _Screen.Caption)
	DelDirectory(lcBaseDirectory)
	DelDirectory(lcDBCDirecotoryOriginal)
	DelDirectory(lcTempDatabaseDirectory)
	RETURN .F.
ENDIF

* 7. 
TRY
	OPEN DATABASE (lcTempDatabaseDirectory + lcDatabaseName) EXCLUSIVE	
CATCH TO loError
	lnErrorNo = loError.ErrorNo
ENDTRY			
IF lnErrorNo > 0
	WAIT CLEAR 
	MESSAGEBOX("Cannot open database exclusive. Operation canceled.", 0 + 16, _Screen.Caption)
	DelDirectory(lcBaseDirectory)
	DelDirectory(lcDBCDirecotoryOriginal)
	DelDirectory(lcTempDatabaseDirectory)
	RETURN .F.
ENDIF 
			
ADBOBJECTS(laTable, "TABLE") 
lnMaxItems = ALEN(laTable)

* For each table
FOR lnItem = 1 TO lnMaxItems

	lcTableName = FORCEEXT(ALLTRIM(laTable[lnItem]), "dbf")
	lcCurrentTable = lcTableName
	
	WAIT WINDOW 'Database repair processing ' + juststem(lcCurrentTable) + '...' NOWAIT NOCLEAR 

	llRepairDBF = .f.
	llRepairCDX = .f.
	llRepairFPT = .f.
	lnMaxErrorCount = 3

	lcBaseDBF = ADDBS(lcBaseDirectory) + FORCEEXT(lcTableName, "dbf")
	lcBaseDBFCDX = FORCEEXT(lcBaseDBF, "cdx")
	IF ADIR(laDummy, lcBaseDBFCDX) = 0
		llRepairCDX = .T.
	ENDIF
	lcBaseDBFFPT = FORCEEXT(lcBaseDBF, "fpt")
	IF ADIR(lDummy, lcBaseDBFFPT) = 0
		llRepairFPT = .T.
	ENDIF
	IF ADIR(laDummy, lcTempDatabaseDirectory + FORCEEXT(lcTableName, "dbf")) = 1
		lnErrorNo = 0
		IF ADIR(laDummy, lcTempDatabaseDirectory  + FORCEEXT(lcTableName, "cdx")) = 0 AND ADIR(laDummy, lcBaseDBFCDX) = 1
			TRY
				COPY FILE (lcBaseDBFCDX) TO (lcTempDatabaseDirectory + FORCEEXT(lcTableName, "cdx"))
			CATCH
				This.cCurrentTable = ""
			ENDTRY
		ENDIF
				
		DO WHILE (llRepairDBF = .F. OR llRepairCDX = .F. OR llRepairFPT = .F.) AND lnMaxErrorCount > 0
			* 8.1.
			IF llRepairDBF = .F.
				llRepairDBF = .T.
				IF !RepairFileHeader(lcTempDatabaseDirectory + lcTableName, lcBaseDirectory + lcTableName)
					lcCurrentTable = ""
					STORE .T. TO llRepairDBF, llRepairCDX, llRepairFPT
				ENDIF
			ENDIF

			lnErrorNo = 0
			TRY 
				USE (lcTempDatabaseDirectory + lcTableName) ALIAS TestTable EXCLUSIVE
			CATCH TO loError
				lnErrorNo = loError.ErrorNo
			ENDTRY
			IF USED("TestTable")
				USE IN TestTable
			ENDIF

			DO CASE
				* 8.2.
				* Index does not match the table.
				CASE lnErrorNo = 114 
					IF !llRepairCDX
						llRepairCDX = .T.
						IF ADIR(laDummy, lcBaseDBFCDX) = 1 
							TRY
								COPY FILE (lcBaseDBFCDX) TO (lcTempDatabaseDirectory + FORCEEXT(lcTableName, "cdx"))
								USE (lcTempDatabaseDirectory + lcTableName) ALIAS TestTable EXCLUSIVE
								SELECT TestTable
								DELETE TAG ALL
								USE IN TestTable
							CATCH
							ENDTRY
							IF USED("TestTable")
								USE IN TestTable
							ENDIF
						ENDIF
					ELSE
						STORE .T. TO llRepairDBF, llRepairFPT 
					ENDIF

				* 8.3.
				* Memo file "name" is missing or is invalid.
				CASE lnErrorNo = 41 
					IF !llRepairFPT
						llRepairFPT = .T.
						IF ADIR(laDummy, lcTempDatabaseDirectory + FORCEEXT(lcTableName, "fpt")) = 1
							* Try replace memo header
							lnFileHandleNumber1 = 0
							lnFileHandleNumber2 = 0
							TRY
								lnFileHandleNumber1 = FOPEN(lcTempDatabaseDirectory  + FORCEEXT(lcTableName, "fpt"), 12) 
								lnFileHandleNumber2 = FOPEN(lcBaseDBFFPT, 10)
								IF lnFileHandleNumber1 > 0 AND lnFileHandleNumber2 > 0
									FWRITE(lnFileHandleNumber1, FREAD(lnFileHandleNumber2, 512))
									FCLOSE(lnFileHandleNumber1)
									lnFileHandleNumber1 = 0
									FCLOSE(lnFileHandleNumber2)
									lnFileHandleNumber2 = 0
									TRY
										USE (lcTempDatabaseDirectory + lcTableName) AGAIN ALIAS TestTable EXCLUSIVE
										llReplaceMemoFile = .F.
									CATCH TO loError
										* Memo file "name" is missing or is invalid.
										IF loError.ErrorNo = 41 
											llReplaceMemoFile = .T.
										ENDIF
									ENDTRY
								ELSE
									llReplaceMemoFile = .T.
								ENDIF
							CATCH
								llReplaceMemoFile = .T.
							FINALLY
								IF lnFileHandleNumber1 > 0
									FCLOSE(lnFileHandleNumber1)
								ENDIF
								IF lnFileHandleNumber2 > 0
									FCLOSE(lnFileHandleNumber2)
								ENDIF
								IF USED("TestTable")
									USE IN TestTable 
								ENDIF
							ENDTRY
						ELSE
							llReplaceMemoFile = .T.
						ENDIF
						IF llReplaceMemoFile
							TRY
								COPY FILE (lcBaseDBFFPT) TO (lcTempDatabaseDirectory + FORCEEXT(lcTableName, "fpt"))
								USE (lcBaseDBF) IN 0 AGAIN ALIAS _GetTableStructure SHARED
								lnFieldCount = AFIELDS(laFieldList, "_GetTableStructure")
								USE IN _GetTableStructure 
								lcDropMemo = ""
								FOR lnCurrentField = 1 TO lnFieldCount
									IF laFieldList[lnCurrentField, 2] $ "MGW"
										lcDropMemo = lcDropMemo + " DROP COLUMN " + ALLTRIM(laFieldList[lnCurrentField, 1])
									ENDIF
								ENDFOR
								IF !EMPTY(lcDropMemo)
									lcDropMemo = "ALTER TABLE [" + lcTempDatabaseDirectory + lcTableName + "] " + lcDropMemo
									TRY
										&lcDropMemo
									CATCH
										lcCurrentTable = ""
									ENDTRY
									IF USED(JUSTSTEM(lcTableName))
										USE IN (JUSTSTEM(lcTableName))
									ENDIF
								ENDIF
							CATCH
								lcCurrentTable = ""
							ENDTRY
						ENDIF
					ELSE
						STORE .T. TO llRepairDBF, llRepairCDX 
					ENDIF

				* File is in use.
				CASE lnErrorNo = 3 
					STORE .T. TO llRepairDBF, llRepairCDX, llRepairFPT
					* Do Nothing
					lcCurrentTable = ""
					
				OTHERWISE
					IF USED("TestTable")
						USE IN TestTable 
					ENDIF
					lnMaxErrorCount = lnMaxErrorCount - 1
					IF lnMaxErrorCount = 0 AND lnErrorNo > 0
						lcCurrentTable = ""
					ENDIF
					IF lnErrorNo = 0
						lnMaxErrorCount = 0
					ENDIF
			ENDCASE
			IF USED("TestTable")
				USE IN TestTable
			ENDIF
		ENDDO

		* 8.4.; 8.5.
		IF !ClearCorruptedRecords(lcTempDatabaseDirectory + lcTableName)
			lcCurrentTable = ""
		ENDIF
	
		IF FILE(lcBaseDirectory + lcTableName)
			DIMENSION laAutoIncFields[1,3]
			* 8.6.; 8.7.
			IF !TransferData(lcTempDatabaseDirectory + lcTableName, lcBaseDBF, @laAutoIncFields)
				lcCurrentTable = ""
			ENDIF 

			FOR lnI = 1 TO ALEN(laAutoIncFields, 1)
				IF ALEN(laAutoIncs, 1) = 1 AND EMPTY(laAutoIncs[1, 1])
					lnRow = 1
				ELSE 
					lnRow = ALEN(laAutoIncs, 1) + 1
				ENDIF 	
				DIMENSION laAutoIncs[lnRow, 4]
				laAutoIncs[lnRow, 1] = laAutoIncFields[lnI, 1]	&& next value
				laAutoIncs[lnRow, 2] = laAutoIncFields[lnI, 2]	&& step
				laAutoIncs[lnRow, 3] = laAutoIncFields[lnI, 3]	&& field name
				laAutoIncs[lnRow, 4] = lcTableName
			ENDFOR 	
		ENDIF
			
		* Occur error
		IF EMPTY(lcCurrentTable) 
			lcNotRepairTables = lcNotRepairTables + lcComma + lcTableName
			lcComma = ","
			laTable[lnItem] = ""
		ENDIF			
	ENDIF	
	IF USED(lcTableName)
		USE IN (lcTableName)
	ENDIF
NEXT

* 9.
* Repair
WAIT WINDOW 'Database repair process is finishing...' NOWAIT NOCLEAR 

IF DBUSED(ADDBS(tcPath) + lcDatabaseName)
	SET DATABASE TO (ADDBS(tcPath) + lcDatabaseName)
	CLOSE DATABASES
ENDIF
IF DBUSED(lcBaseDirectory + lcDatabaseName)
	SET DATABASE TO (lcBaseDirectory + lcDatabaseName)
	CLOSE DATABASES
ENDIF
IF DBUSED(lcTempDatabaseDirectory + lcDatabaseName)
	SET DATABASE TO (lcTempDatabaseDirectory + lcDatabaseName)
	CLOSE DATABASES
ENDIF

* 10.
* Entire database
TRY
	lcFrom = lcDBCDirecotoryOriginal + lcDatabaseName
	lcTo = ADDBS(lcBaseDirectory) + lcDatabaseName
	COPY FILE (lcFrom) TO (lcTo)
	COPY FILE (FORCEEXT(lcFrom, "dct")) TO (FORCEEXT(lcTo, "dct"))
	COPY FILE (FORCEEXT(lcFrom, "dcx")) TO (FORCEEXT(lcTo, "dcx"))
CATCH TO loError
	lnErrorNo = loError.ErrorNo
ENDTRY

* 11.
* Set Next AutoIncs
OPEN DATABASE (ADDBS(lcBaseDirectory) + lcDatabaseName) EXCLUSIVE 
lcCurrTable = ""
FOR lnCurrentRow = 1 TO ALEN(laAutoIncs, 1)
	lnLastAutoIncValue = 0
	IF ALLTRIM(LOWER(JUSTSTEM(laAutoIncs[lnCurrentRow, 4]))) == ALLTRIM(LOWER(lcCurrTable))
		SELECT _NewTable
	ELSE 
		IF USED("_NewTable")
			USE IN _NewTable
		ENDIF	
		lcCurrTable = JUSTSTEM(laAutoIncs[lnCurrentRow, 4])
		USE (lcCurrTable) ALIAS _NewTable EXCLUSIVE 
	ENDIF 	
	lcFieldName = laAutoIncs[lnCurrentRow, 3]
	CALCULATE MAX(&lcFieldName.) TO lnLastAutoIncValue	
	lnLastAutoIncValue = NVL(lnLastAutoIncValue, 0) + laAutoIncs[lnCurrentRow, 2]
	lnLastAutoIncValue = MAX(lnLastAutoIncValue, laAutoIncs[lnCurrentRow, 1])

	lcBuffer = "ALTER TABLE [" + JUSTSTEM(laAutoIncs[lnCurrentRow, 4]) + "] " + ;
						"ALTER COLUMN " + laAutoIncs[lnCurrentRow, 3] + " Integer " + ;
						" AUTOINC NEXTVALUE " + TRANSFORM(lnLastAutoIncValue) + ;
						" STEP " + TRANSFORM(laAutoIncs[lnCurrentRow, 2])
	TRY
		&lcBuffer.
	CATCH
		*
	ENDTRY
ENDFOR

* 12.
CLOSE TABLES ALL
IF DBUSED(lcBaseDirectory + lcDatabaseName)
	SET DATABASE TO (lcBaseDirectory + lcDatabaseName)
	CLOSE DATABASES
ENDIF

* 13.
lnTableCount = ALEN(laTable)
FOR lnItem = 1 TO lnTableCount
	lcTableName = ALLTRIM(laTable[lnItem])
	* Error occur. No copy dbf.
	IF EMPTY(lcTableName) 
		LOOP
	ENDIF
	lcTableName = FORCEEXT(lcTableName, "dbf")
	lcTableNameCDX = FORCEEXT(lcTableName, "cdx")
	lcTableNameFPT = FORCEEXT(lcTableName, "fpt")
	TRY
		IF FILE(lcBaseDirectory + lcTableName)
			COPY FILE (lcBaseDirectory + lcTableName) TO (ADDBS(tcPath) + lcTableName)
		ENDIF
		IF FILE(lcBaseDirectory + lcTableNamecdx)
			COPY FILE (lcBaseDirectory + lcTableNamecdx) TO (ADDBS(tcPath) + lcTableNamecdx)
		ENDIF
		IF FILE(lcBaseDirectory + lcTableNamefpt)
			COPY FILE (lcBaseDirectory + lcTableNamefpt) TO (ADDBS(tcPath) + lcTableNamefpt)
		ENDIF
	CATCH TO loError
		lnErrorNo = loError.ErrorNo
	ENDTRY
ENDFOR
*--- Copy database-container files
lcFrom = lcBaseDirectory + lcDatabaseName
lcTo = ADDBS(tcPath) + lcDatabaseName
COPY FILE (lcFrom) TO (lcTo)
COPY FILE (FORCEEXT(lcFrom, "dct")) TO (FORCEEXT(lcTo, "dct"))
COPY FILE (FORCEEXT(lcFrom, "dcx")) TO (FORCEEXT(lcTo, "dcx"))

* 14.
DelDirectory(lcTempDatabaseDirectory)
* 15.
DelDirectory(lcBaseDirectory)
* 16.
DelDirectory(lcDBCDirecotoryOriginal)

WAIT CLEAR 

ENDFUNC 

*-------------------------------------------------------
* Function....: CreateEmptyDBC
* Called by...:
* Abstract....:
*
* Returns.....:
*
* Parameters..: tcDBCName, tcTargetFolder (by Ref), 
*								tcRelativePathToKRTFile AS STRING, tcLogFileName AS STRING
* Notes.......: Creates an empty DBC using files generated by GenDBC
*-------------------------------------------------------
FUNCTION CreateEmptyDBC
	LPARAMETERS tcDBCName AS STRING, tcTargetFolder AS STRING, tcRelativePathToKRTFile AS STRING, tcLogFileName AS STRING

	LOCAL llRes AS Logical, loError AS EXCEPTION, lcGenDBCFunction, lcKRTDBFName, lcKRTFile, lcOldSetPath AS STRING, ;
				lnErrorNo AS INTEGER, lcErrorMsg, lcOldDir, lcOrigDBCFunc
	
	lcErrorMsg = ""
	IF EMPTY(tcDBCName) OR VARTYPE(tcDBCName) <> "C"
		RETURN .F.
	ENDIF
	tcDBCName = FORCEEXT(tcDBCName, "dbc")
	IF EMPTY(tcTargetFolder) OR VARTYPE(tcTargetFolder) <> "C"
		tcTargetFolder = ADDBS(SYS(2023)) + SYS(2015)
	ENDIF
	tcTargetFolder = ADDBS(tcTargetFolder)
	IF !DIRECTORY(tcTargetFolder, 1)
		TRY
			MD (tcTargetFolder)
		CATCH TO loError
			lcErrorMsg = loError.MESSAGE
		ENDTRY
	ENDIF
	IF !DIRECTORY(tcTargetFolder, 1)
		RETURN .F.
	ENDIF

	IF VERSION(2) = 2
		lcGenDBCFunction = 'VFX_' + JUSTSTEM(tcDBCName)
	ELSE
		lcGenDBCFunction = SYS(2015)
	ENDIF

	lcKRTFile = ADDBS(SYS(2023)) + 'VFX_' + FORCEEXT(tcDBCName, "krt")
	lcKRTDBFName = 'VFX_' + JUSTSTEM(tcDBCName) + "krt.dbf"
	lcOldDir = GetDefaultFolder()
	lnErrorNo = 0
	lcOldSetPath = SET("Path")
	TRY
		IF !FILE(lcKRTDBFName) AND VERSION(2) = 2
			IF EMPTY(tcRelativePathToKRTFile)
				IF FILE(ADDBS(lcOldDir) + "Data\" + lcKRTDBFName)
					lcKRTDBFName = ADDBS(lcOldDir) + "Data\" + lcKRTDBFName
				ENDIF
			ELSE
				IF FILE(ADDBS(ADDBS(lcOldDir) + tcRelativePathToKRTFile) + lcKRTDBFName)
					lcKRTDBFName = ADDBS(ADDBS(lcOldDir) + tcRelativePathToKRTFile) + lcKRTDBFName
				ENDIF
			ENDIF
		ENDIF
		
		IF FILE(lcKRTDBFName)
			USE (lcKRTDBFName) AGAIN SHARED ALIAS _krtDbf
			SELECT _krtDbf
			LOCATE
			COPY MEMO _krtDbf.PROGRAM TO (lcKRTFile)
			USE IN _krtDbf
		ENDIF
		CD (tcTargetFolder)
		SET PATH TO (JUSTPATH(lcKRTFile)) ADDITIVE
		IF VERSION(2) = 2 AND !FILE(FORCEEXT(lcGenDBCFunction, "prg"))
			IF EMPTY(tcRelativePathToKRTFile)
				IF FILE(ADDBS(lcOldDir) + "Data\" + FORCEEXT(lcGenDBCFunction, "prg"))
					lcGenDBCFunction = ADDBS(lcOldDir) + "Data\" + FORCEEXT(lcGenDBCFunction, "prg")
				ENDIF
			ELSE
				IF FILE(ADDBS(ADDBS(lcOldDir) + tcRelativePathToKRTFile) + FORCEEXT(lcGenDBCFunction, "prg"))
					lcGenDBCFunction = ADDBS(ADDBS(lcOldDir) + tcRelativePathToKRTFile) + FORCEEXT(lcGenDBCFunction, "prg")
				ENDIF
			ENDIF
		ELSE
			lcGenDBCFunction = ADDBS(SYS(2023)) + FORCEEXT(lcGenDBCFunction, "prg")
			lcOrigDBCFunc = FORCEEXT('VFX_' + JUSTFNAME(tcDBCName),"PRG")
			STRTOFILE(FILETOSTR(lcOrigDBCFunc), lcGenDBCFunction)
		ENDIF
		COMPILE (lcGenDBCFunction)

		IF EMPTY(tcLogFileName)
			tcLogFileName = "CreateEmptyDbc.log"
		ENDIF
		DO (lcGenDBCFunction) WITH tcLogFileName
		IF TYPE("_GenDBC_Error") == "L" AND _GenDBC_Error
			llRes = .F.
		ELSE
			llRes = .T.
		ENDIF
	CATCH TO loError
		lcErrorMsg = loError.MESSAGE
	FINALLY
		CD (lcOldDir)
	ENDTRY

	IF DBUSED(tcTargetFolder + tcDBCName)
		SET DATABASE TO (tcTargetFolder + tcDBCName)
		CLOSE DATABASES
	ENDIF
	TRY
		ERASE (lcKRTFile)
		IF VERSION(2) != 2
			lcGenDBCFunction = FORCEEXT(lcGenDBCFunction, "*")
			ERASE (lcGenDBCFunction)
		ENDIF
	CATCH
	ENDTRY

	TRY
		SET PATH TO (lcOldSetPath)
	CATCH
	ENDTRY

	RETURN llRes
ENDFUNC 

*-------------------------------------------------------
* Function....: RepairFileHeader 
* Called by...:
* Abstract....:
*
* Returns.....:
*
* Parameters..: tcCorruptedFileName, lcBaseFileName
*
* Notes.......: Repair File header by copying it from correct source
*-------------------------------------------------------
FUNCTION RepairFileHeader 
	LPARAMETERS tcCorruptedFileName, tcBaseFileName

	LOCAL lnFileSize, lnHandleCorruptedFile, lnHandleBaseFile, lcPositionOfFirstDataRecord, ;
				lcHexPositionOfFirstDataRecord, lnPositionOfFirstDataRecord, lnFieldNumber, lnPosHeaderRecordTerminator, ;
				lnFileHeaderInfoLen, lcBaseFileHeader, lcCorruptedFileHeader, lcSizeOfTableRecord, lcBuffer, ;
				lnSizeOfTableRecord, lnRowCount, lcHexRowCount, lcRowCount, lcNewFileHeader, lnField, lnAutoincDataPos

	* Get File size
	=ADIR(laFileInfo, tcCorruptedFileName) 
	lnFileSize = laFileInfo[2]

	lnHandleCorruptedFile = FOPEN(tcCorruptedFileName, 12)

	IF lnHandleCorruptedFile < 1
		* Cannot open file
		RETURN .F.
	ENDIF

	lnHandleBaseFile = FOPEN(tcBaseFileName, 10)
	IF lnHandleBaseFile < 1
		* Cannot open file
		FCLOSE(lnHandleCorruptedFile)
		RETURN .F.
	ENDIF

	* Get Field number in Base DBF
	FSEEK(lnHandleBaseFile, 8)
	lcPositionOfFirstDataRecord = FREAD(lnHandleBaseFile, 2)
	lcHexPositionOfFirstDataRecord = "0x" + RIGHT(TRANSFORM(ASC(RIGHT(lcPositionOfFirstDataRecord, 1)), "@0"), 2)
	lcHexPositionOfFirstDataRecord = lcHexPositionOfFirstDataRecord + RIGHT(TRANSFORM(ASC(LEFT(lcPositionOfFirstDataRecord, 1)), "@0"), 2)
																
	lnPositionOfFirstDataRecord = VAL(lcHexPositionOfFirstDataRecord)
	lnFieldNumber = (lnPositionOfFirstDataRecord - 296)/32

	lnPosHeaderRecordTerminator = 32 + (lnFieldNumber * 32) + 1
	lnFileHeaderInfoLen = lnPosHeaderRecordTerminator + 263

	* Go to file beginning
	FSEEK(lnHandleBaseFile, 0) 

	* Get header records
	lcBaseFileHeader = FREAD(lnHandleBaseFile, lnFileHeaderInfoLen)
	lcCorruptedFileHeader = FREAD(lnHandleCorruptedFile, lnFileHeaderInfoLen)

	* Get size (width) of a table record.
	lcSizeOfTableRecord = SUBSTR(lcBaseFileHeader, 11, 2)
	lcBuffer = "0x" + RIGHT(TRANSFORM(ASC(RIGHT(lcSizeOfTableRecord, 1)), "@0"), 2) + RIGHT(TRANSFORM(ASC(LEFT(lcSizeOfTableRecord, 1)), "@0"), 2)
	lnSizeOfTableRecord = VAL(lcBuffer)

	* Get table row count
	lnRowCount = ROUND((lnFileSize - lnFileHeaderInfoLen - 1) / lnSizeOfTableRecord, 0)
	lcHexRowCount = TRANSFORM(lnRowCount, "@0")
	lcRowCount = CHR(VAL("0x" + RIGHT(lcHexRowCount, 2)))
	lcRowCount = lcRowCount + CHR(VAL("0x" + SUBSTR(lcHexRowCount, 7, 2)))
	lcRowCount = lcRowCount + CHR(VAL("0x" + SUBSTR(lcHexRowCount, 5, 2)))
	lcRowCount = lcRowCount + CHR(VAL("0x" + SUBSTR(lcHexRowCount, 3, 2)))		

	lcNewFileHeader = STUFF(lcBaseFileHeader, 5, 4, lcRowCount)

	* Get current Autoinc values and replace them in new header record
	FOR lnField = 1 TO lnFieldNumber 
		lnAutoincDataPos = (lnField * 32) + ;	&& 32 + ((lnFieldNumber -1)*32)
											20	&& 0-Based:
												&& 19 - 22  Value of autoincrement Next value  
												&& 23  Value of autoincrement Step value  
													
		lcNewFileHeader = STUFF(lcNewFileHeader, lnAutoincDataPos, 3, ;
								SUBSTR(lcCorruptedFileHeader, lnAutoincDataPos, 3))
	ENDFOR

	* Go to file beginning
	FSEEK(lnHandleCorruptedFile, 0, 0) 

	FWRITE(lnHandleCorruptedFile, lcNewFileHeader)

	FSEEK(lnHandleCorruptedFile, lnPositionOfFirstDataRecord)

	FCLOSE(lnHandleCorruptedFile)
	FCLOSE(lnHandleBaseFile)
ENDFUNC 

*-------------------------------------------------------
* Function....: ClearCorruptedRecords 
* Called by...:
* Abstract....:
*
* Returns.....:
*
* Parameters..: tcDBFFileName
*
* Notes.......: Deletes corrupted records in a table
*-------------------------------------------------------
FUNCTION ClearCorruptedRecords 
	LPARAMETERS tcDBFFileName

	LOCAL lnErrorNo, loError as Exception, lnTagNumber, lcPrimaryExpression, lnKk, lcTagType, lcTagExpression, ;
				lnFieldsNumber, lcExpressionDelRow, lcOperator, llEvalDelExpr, lnNewRecno, llPrimaryExpr, laTag[1], ;
				laFields[1], llRecordIsDeleted

	lnErrorNo = 0
	TRY
		USE (tcDBFFileName) ALIAS _CorruptedTable EXCLUSIVE
		DELETE TAG ALL
		IF !EMPTY(CURSORGETPROP("Database"))
			DELETE TRIGGER ON (DBF("_CorruptedTable")) FOR DELETE
			DELETE TRIGGER ON (DBF("_CorruptedTable")) FOR INSERT
			DELETE TRIGGER ON (DBF("_CorruptedTable")) FOR UPDATE
		ENDIF
	CATCH TO loError
		lnErrorNo = loError.ErrorNo
	ENDTRY
	IF lnErrorNo > 0
		RETURN .F.
	ENDIF
	SELECT _CorruptedTable
	lnTagNumber = ATAGINFO(laTag)
	lcPrimaryExpression = ""
	FOR lnKk = 1 TO lnTagNumber
		lcTagType = LOWER(laTag[lnKk, 2])
		IF lcTagType == "primary" OR ; && Tag Type
			lcTagType = "candidate"
			lcTagExpression = laTag[lnKk,3]
			
			SELECT _CorruptedTable
			SCAN 
				IF EMPTY(EVALUATE(lcTagExpression))
					DELETE 				
					llRecordIsDeleted = .T.
				ENDIF 
			ENDSCAN
			 	
			IF lcTagType == "primary"
				lcPrimaryExpression = lcTagExpression
			ENDIF
		ENDIF
	ENDFOR
	lnFieldsNumber = AFIELDS(laFields)
	lcExpressionDelRow = ""
	lcOperator = ""
	FOR lnKk = 1 TO lnFieldsNumber
		IF INLIST(laFields[lnKk,2], "C", "M")
			lcExpressionDelRow = lcExpressionDelRow + lcOperator + ;
			"At(CHR(0), " + laFields[lnKk,1] + ") > 0"
			lcOperator = " or "
		ENDIF
	ENDFOR

	LOCATE

	IF !EMPTY(lcExpressionDelRow) OR !EMPTY(lcPrimaryExpression)
		SCAN 
			llEvalDelExpr = .T.
			lvPrimaryExpression = ""
			TRY 
				IF EMPTY(lcExpressionDelRow)
					llEvalDelExpr = .F.
				ELSE 
					llEvalDelExpr = EVALUATE(lcExpressionDelRow)
				ENDIF 
			CATCH 
				llEvalDelExpr = .T.
			ENDTRY 
			
			IF llEvalDelExpr 
				DELETE		
				llRecordIsDeleted = .T.
			ELSE
				IF !EMPTY(lcPrimaryExpression)
					TRY 
						lvPrimaryExpression = EVALUATE(lcPrimaryExpression)
					CATCH 
						lvPrimaryExpression = ""
					ENDTRY 
				ENDIF
				EXIT
		 	ENDIF
		ENDSCAN
		lnNewRecno = RECNO() + 1

		IF RECCOUNT() >= lnNewRecno
			GO lnNewRecno 
			SCAN REST
				llEvalDelExpr = .T.
				llPrimaryExpr = .T.
				TRY 
					IF EMPTY(lcExpressionDelRow)
						llEvalDelExpr = .F.
					ELSE 
						llEvalDelExpr = EVALUATE(lcExpressionDelRow)
					ENDIF 
				CATCH 
					llEvalDelExpr = .T.
				ENDTRY 
				
				IF llEvalDelExpr 
					DELETE
					llRecordIsDeleted = .T.
				ELSE
					IF !EMPTY(lcPrimaryExpression)
						TRY 
							llPrimaryExpr = (lvPrimaryExpression  = EVALUATE(lcPrimaryExpression))
						CATCH 
							llPrimaryExpr = .T.
						ENDTRY 
						IF llPrimaryExpr
							DELETE
							llRecordIsDeleted = .T.
						ELSE
							TRY 
								lvPrimaryExpression  = EVALUATE(lcPrimaryExpression)
							CATCH 
								lvPrimaryExpression  = ""
							ENDTRY 
						ENDIF
					ENDIF
				ENDIF
			ENDSCAN
		ENDIF
	ENDIF
	
	IF USED("_CorruptedTable")
		IF llRecordIsDeleted 
			PACK 
		ELSE 
			REINDEX
		ENDIF 	
		USE IN _CorruptedTable 
	ENDIF

ENDFUNC 

*-------------------------------------------------------
* Function....: TransferData
* Called by...:
* Abstract....:
*
* Returns.....:
*
* Parameters..: tcOldTable, tcNewTable, taAutoIncFields
*
* Notes.......: Transfer data from tcOldTable to tcNewTable
*-------------------------------------------------------
FUNCTION TransferData
	LPARAMETERS tcOldTable, tcNewTable, taAutoIncFields
	
	LOCAL llSuccess, lnSetDel, lcDatabaseFileName, lnFieldCount, lnCurrentField, ;
				lnAutoIncFieldsCount, lnAutoIncNewFieldsCount, lnNewFieldCount, lcCurrentFieldName, ;
				lnNewFieldPos, loError, lnErrorNo, llUpdateError, loRowContent, llDeleted
	LOCAL ARRAY laAutoIncFields[1, 3]
			
	llSuccess = .T.

	lnSetDel = SET("Deleted")
	SET DELETED OFF

	SELECT 0
	USE (tcNewTable) EXCLUSIVE ALIAS _NewTable
	lcDatabaseFileName = CURSORGETPROP("Database")
	IF !EMPTY(lcDatabaseFileName)
		SET DATABASE TO (lcDatabaseFileName)
		DELETE TRIGGER ON (tcNewTable) FOR DELETE
		DELETE TRIGGER ON (tcNewTable) FOR INSERT
		DELETE TRIGGER ON (tcNewTable) FOR UPDATE
		IF !EMPTY(DBGETPROP(JUSTSTEM(tcNewTable), "TABLE", "RuleExpression"))
			ALTER TABLE (tcNewTable) DROP CHECK
		ENDIF
		lnFieldCount = AFIELDS(laNewFieldList, "_NewTable")
		FOR lnCurrentField = 1 TO lnFieldCount
			* Field validation expression
			IF !EMPTY(laNewFieldList[lnCurrentField, 7]) 
				ALTER TABLE (tcNewTable) ALTER COLUMN (laNewFieldList[lnCurrentField, 1]) DROP CHECK
			ENDIF
			* Field default value
			IF !EMPTY(laNewFieldList[lnCurrentField, 9]) 
				ALTER TABLE (tcNewTable) ALTER COLUMN (laNewFieldList[lnCurrentField, 1]) DROP DEFAULT
			ENDIF
		ENDFOR
	ENDIF
	
	CURSORSETPROP("Buffering", 3, "_NewTable")
	
	* Check for Integer Autoinc
	lnAutoIncFieldsCount = 0
	lnAutoIncNewFieldsCount = 0
	TRY
		USE (tcOldTable) IN 0 ALIAS _OldTable
		lnFieldCount = AFIELDS(laFieldList, "_OldTable")
		lnNewFieldCount = AFIELDS(laNewFieldList, "_NewTable")
		lcCurrentFieldName = ""
		FOR lnCurrentField = 1 TO lnFieldCount
			IF laFieldList[lnCurrentField, 2] == "I"
				lcCurrentFieldName = laFieldList[lnCurrentField,1]
				* Fields in two arrays may be in different positions
				lnNewFieldPos = ASCAN(laNewFieldList, lcCurrentFieldName, 1, lnNewFieldCount, 1, 15)
				IF lnNewFieldPos > 0		
					* field is present in both old & new table
					DO CASE
						CASE laFieldList[lnCurrentField, 18] = laNewFieldList[lnNewFieldPos, 18]
							IF laFieldList[lnCurrentField, 18] <> 0
								ALTER TABLE (tcNewTable) ALTER COLUMN (laFieldList[lnCurrentField, 1]) INTEGER
								
								lnAutoIncFieldsCount = lnAutoIncFieldsCount + 1
								DIMENSION laAutoIncFields[lnAutoIncFieldsCount, 3]
								laAutoIncFields[lnAutoIncFieldsCount, 1] = NVL(laFieldList[lnCurrentField, 17], 0)	&& next value
								laAutoIncFields[lnAutoIncFieldsCount, 2] = NVL(laFieldList[lnCurrentField, 18], 0)	&& step
								laAutoIncFields[lnAutoIncFieldsCount, 3] = laFieldList[lnCurrentField, 1]
							ENDIF
							
						CASE laFieldList[lnCurrentField, 18] = 0 AND laNewFieldList[lnNewFieldPos, 18] <> 0
							* field in old table is Int, in new table is Int(AutoInc)
							ALTER TABLE (tcNewTable) ALTER COLUMN (laFieldList[lnCurrentField, 1]) INTEGER
							
							lnAutoIncFieldsCount = lnAutoIncFieldsCount + 1
							DIMENSION laAutoIncFields[lnAutoIncFieldsCount, 3]
							laAutoIncFields[lnAutoIncFieldsCount, 1] = NVL(laNewFieldList[lnNewFieldPos, 17], 0)	&& next value
							laAutoIncFields[lnAutoIncFieldsCount, 2] = NVL(laNewFieldList[lnNewFieldPos, 18], 0)	&& step
							laAutoIncFields[lnAutoIncFieldsCount, 3] = laFieldList[lnCurrentField, 1]
							
						CASE laFieldList[lnCurrentField, 18] <> 0 AND laNewFieldList[lnNewFieldPos, 18] = 0
							* field in old table is Int(AutoInc), in new table is Int
							* Do nothing

						OTHERWISE
							* both fields AutoInc, different increment step - use increment step from new table
							ALTER TABLE (tcNewTable) ALTER COLUMN (laFieldList[lnCurrentField, 1]) INTEGER
							
							lnAutoIncFieldsCount = lnAutoIncFieldsCount + 1
							DIMENSION laAutoIncFields[lnAutoIncFieldsCount, 3]
							laAutoIncFields[lnAutoIncFieldsCount, 1] = NVL(laFieldList[lnCurrentField, 17], 0)	&& next value
							laAutoIncFields[lnAutoIncFieldsCount, 2] = NVL(laNewFieldList[lnNewFieldPos, 18], 0)	&& step
							laAutoIncFields[lnAutoIncFieldsCount, 3] = laFieldList[lnCurrentField, 1]
					ENDCASE
				ENDIF
			ENDIF
		ENDFOR
	CATCH TO loError
		lnAutoIncFieldsCount  = 0
	FINALLY
		IF USED("_OldTable")
			USE IN _OldTable
		ENDIF
	ENDTRY

	* append data row-by-row
	* used by database repair
	lnErrorNo = 0
	TRY
		USE (tcOldTable) AGAIN ALIAS _OldTable IN 0 SHARED
	CATCH TO loError
		llSuccess = .F.
		lnErrorNo = loError.ErrorNo
	ENDTRY

	IF USED("_OldTable")
		SELECT _OldTable
		SCAN
			llUpdateError = .F.
			loRowContent = .NULL.
			lnErrorNo = 0
			llDeleted = DELETED()
			TRY
				SCATTER MEMO NAME loRowContent
			CATCH TO loError
				lnErrorNo = loError.ErrorNo
				loRowContent = .NULL.
			ENDTRY
			IF lnErrorNo > 0
				TRY
					SCATTER NAME loRowContent
				CATCH TO loError
					lnErrorNo = loError.ErrorNo
					loRowContent = .NULL.
				ENDTRY
			ENDIF
			IF !ISNULL(loRowContent)
				TRY
					SELECT _NewTable
					APPEND BLANK
					GATHER NAME loRowContent MEMO
					IF llDeleted
						DELETE
					ENDIF
					* Flag llUpdateError added, because when TableUpdate returns .F., the Catch is not raised
					IF !TABLEUPDATE(.F.)
						llUpdateError = .T.
					ENDIF
				CATCH
					llUpdateError = .T.
				ENDTRY
				IF llUpdateError
					 = TABLEREVERT(.F.)
				ENDIF
			ENDIF
		ENDSCAN
	ENDIF

	IF !USED("_OldTable")
		USE (tcOldTable) IN 0 ALIAS _OldTable
	ENDIF
	IF lnAutoIncFieldsCount > 0 
			* Called from Database Repair Tool 
			* skip Next Autoinc value calculation. Just return the array with autoincs
			ACOPY(laAutoIncFields, taAutoIncFields)
	ENDIF

	USE IN _NewTable
	IF USED("_OldTable")
		USE IN _OldTable
	ENDIF
	SET DELETED &lnSetDel
	RETURN llSuccess
ENDFUNC 

*-------------------------------------------------------
* Function....: GetDefaultFolder()
* Called by...:
*
* Abstract....:
*
* Returns.....:
*
* Parameters..:
*
* Notes.......:
*-------------------------------------------------------
FUNCTION GetDefaultFolder

IF VERSION(2) = 2
     RETURN SYS(5) + SYS(2003)
ELSE
     RETURN JUSTPATH(SYS(16, 0))
ENDIF
ENDFUNC

*-------------------------------------------------------
* Function....: DelDirectory
* Called by...:
*
* Abstract....: Delete Directory
*
* Parameters..:	tcDIR
*
* Notes.......:
*-------------------------------------------------------
FUNCTION DelDirectory(tcDIR)

LOCAL llResult, lnNumDelFiles, lnJ, lnAttemptNum, loError as Exception
LOCAL ARRAY laDelFilesName[1]

TRY
	llResult = .F.

	IF DIRECTORY(tcDIR, 1)
		lnNumDelFiles = ADIR(laDelFilesName, ADDBS(tcDIR) + "*.*", "D")
		* Skip "\." and "\.."
		FOR lnJ = 3 TO lnNumDelFiles	
			IF DIRECTORY(ADDBS(tcDIR) + laDelFilesName[lnJ, 1], 1)
				DelDirectory(ADDBS(tcDIR) + laDelFilesName[lnJ, 1])
			ELSE 	
				ERASE (ADDBS(tcDIR) + laDelFilesName[lnJ, 1])
			ENDIF
		NEXT
		
		lnAttemptNum = 10
		DO WHILE lnAttemptNum > 0
			* try several times, because of timing problems (The directory is not empty.)
			TRY
				RD (tcDIR)
				lnAttemptNum = 0
			CATCH TO loError
				lnAttemptNum = lnAttemptNum - 1
			ENDTRY
		ENDDO
		RELEASE laDelFilesName
		llResult = .T.
	ENDIF
CATCH TO loError
	llResult = .F.
ENDTRY

RETURN llResult

ENDFUNC